#include <vector>
#include <string>
#include <array>
#include <memory>
#include <regex>
#include "../include/correlator.h"
#include "../include/propagator.h"
#include "../include/file_manip.h"

template <size_t N>
void corr_npt<N>::print(std::ostream& os) {
    os << N <<"-point correlator:" << std::endl;
    os << "Gamma structures is: " << gtos(gamma[0]) << " " << gtos(gamma[1]) << " ";
    if
#if __cplusplus >=201703L
      constexpr
#endif
      (N>2){
      os << gtos(gamma[2]);
    }
    os << std::endl;

    os << "source at "<< xn[0] << std::endl;
    if
#if __cplusplus >=201703L
      constexpr
#endif
      (N>2){
      os << "sink at " << xn[1] <<std::endl;
      }
    os << "propagator specification: "<<std::endl;
    for(size_t i=0; i<N;i++){
      os << "propagator id: "<< propagator_id[i]<< std::endl;
      if(is_seq[i])
	os << "Is sequential propagator. Source propagator id: " << seq_id[i] << std::endl;
      os << "\t kappa:" << kappa[i] <<std::endl;
      os << "\t mu:   " << mus[i] << std::endl;
      os << "\t theta:";
      for(auto t :theta[i]) os << t << " ";
      os <<std::endl;
    }
    os << std::endl;
    os << "Gamma structure at source:" << gtos(gamma[0])<< std::endl;
    if
#if __cplusplus >=201703L
     constexpr
#endif
      (N>2)
      {
      for(size_t i=2; i<N; i++)
	os << "Gamma structure injected at: " << xn[i-1] << "  " << gtos(gamma[i]) << std::endl;
      }
  os << "Gamma structure at sink (i.e varying position):  " << gtos(gamma[1]) << std::endl;
}

template<size_t N>

#if __cplusplus >=201402L
std::unique_ptr<corr> corr_npt<N>::make_unique_ptr(){
  return std::make_unique<corr_npt<N>>(*this);
}
#endif

// read a correlator form the input file.
// Some assumption are made:
// - you are at the beginning of the correlator section.
// - the correlator  section is organized as:
//   prop
//   mu
//   theta
//   seq_prop
//   seq_type
//   seq_x0
//
void read_all_corr(std::ifstream &file, std::vector<std::unique_ptr<corr>> &corrs,std::vector<propagator> &prop, int ncorr){
  corrs.clear();
  corrs.resize(ncorr);
  int  npoint;
  std::array<int,3> prop_id;
  std::array<Gamma,3> gamma;
  std::array<double,3> kappa;
  std::array<double,3> mus;
  std::array<bool,3> is_seq;
  std::array<int, 3> seq_id;
  std::array<int,2> xn;// position of the N points.
  std::array<std::array<double,3>,3> theta;
  std::string temp;
  std::regex rgx_int("\\s[+-]?[0-9]+");
  std::regex rgx_gamma("((G[0-5]){1,2})|1");
  std::vector<std::string> aux;

  for(auto i=0; i<ncorr; i++){
    find_section(file, "Correlator "+std::to_string(i));
    npoint = 2;
    read_line(file, temp);
    check_line(temp,"prop");
    aux = get_all_match(temp,rgx_int);
    prop_id[0] = std::stoi(aux[0]);
    prop_id[1] = std::stoi(aux[1]);
    is_seq = {false,false,false};
    if(prop[prop_id[0]].seq_prop){
      is_seq[0] = true;
      npoint++;
      prop_id[2] = prop[prop_id[0]].seq_id;
      seq_id[0] = prop_id[2];
      xn[1] = prop[prop_id[0]].src;
      gamma[2] = prop[prop_id[0]].seq_type;
    }
    if(prop[prop_id[1]].seq_prop){
      is_seq[1] = true;
      npoint++;
      prop_id[2] = prop[prop_id[1]].seq_id;
      seq_id[1] = prop_id[2];
      xn[1] = prop[prop_id[1]].src;
      gamma[2] = prop[prop_id[1]].seq_type;
    }

    read_line(file,temp);
    check_line(temp,"isubprop");

    read_line(file,temp);

    check_line(temp,"type");

    aux = get_all_match(temp,rgx_gamma);

    gamma[0] = stog(aux[0]);
    gamma[1] = stog(aux[1]);

    read_line(file,temp);
    check_line(temp,"gsmear");

    read_line(file,temp);
    check_line(temp,"qsmear");

    read_line(file,temp);
    check_line(temp, "x0");
    xn[0] = std::stoi(get_match(temp,rgx_int)); //
						//
    kappa[0] = prop[prop_id[0]].kappa;
    kappa[1] = prop[prop_id[1]].kappa;
    mus[0] = prop[prop_id[0]].mu;
    mus[1] = prop[prop_id[1]].mu;
    theta[0] = prop[prop_id[0]].theta;
    theta[1] = prop[prop_id[1]].theta;

    if(npoint ==2){

#if __cplusplus>=201402L
      corrs[i]=std::make_unique<corr_npt<2>>(
					     corr_npt<2>(
							 std::array<int,2>{prop_id[0],prop_id[1]},
							 std::array<Gamma,2>{gamma[0],gamma[1]},
							 std::array<double,2>{kappa[0],kappa[1]},
							 std::array<double,2>{mus[0],mus[1]},
							 std::array<int,1>{xn[0]},
							 std::array<std::array<double,3>,2>{theta[0],theta[1]},
							 std::array<bool,2>{false,false},
							 std::array<int,2>{0,0}
							 )
					     );
#else
      corrs[i]= new corr_npt<2>(std::array<int,2>{prop_id[0],prop_id[1]},
				std::array<Gamma,2>{gamma[0],gamma[1]},
				std::array<double,2>{kappa[0],kappa[1]},
				std::array<double,2>{mus[0],mus[1]},
				std::array<int,1>{xn[0]},
				std::array<std::array<double,3>,2>{theta[0],theta[1]},
				std::array<bool,2>{false,false},
				std::array<int,2>{0,0}
				);
#endif
    }
    else if(npoint==3){
      kappa[2] = prop[prop_id[2]].kappa;
      mus[2] = prop[prop_id[2]].mu;
      theta[2] = prop[prop_id[2]].theta;

#if __cplusplus>= 201402L
      corrs[i] = std::make_unique<corr_npt<3>>(corr_npt<3>(prop_id,gamma,kappa,mus,xn,theta,is_seq,seq_id));
#else
      corrs[i] = new corr_npt<3>(prop_id,gamma,kappa,mus,xn,theta,is_seq,seq_id);
#endif
    }

    else if(npoint >3){
      std::cerr << "[read_corr] "<< npoint <<"-point function are not supported yet. "<<std::endl;
      std::cerr << "\t reading correlator " << i << "with propagators " << prop_id[0] << " " << prop_id[1] << std::endl;
      exit(1);
    }
  }
}
bool compare_corr(const corr& A, const corr& B){

  if (A.npoint()!=B.npoint()) return false;

  for(size_t i =0;i<A.npoint() ; i++){
    if(A.get_propagator_id(i) != B.get_propagator_id(i))
      return false;
  };

  return true;
}


#if __cplusplus>=201402L
bool compare_corr(const std::unique_ptr<corr> &A, const std::unique_ptr<corr> &B){
  return compare_corr(*A,*B);
}
#else
bool compare_corr(const corr* A, const corr *B){
   if (A->npoint()!=B->npoint()) return false;

  for(size_t i =0;i < A->npoint() ; i++){
    if(A->get_propagator_id(i) != B->get_propagator_id(i))
      return false;
  };

  return true;
}
#endif

#if __cplusplus>=201402L
void condense_correlator(std::vector<std::unique_ptr<corr>>& corrs){
  std::vector<std::unique_ptr<corr>> condensed;
#else
  void condense_correlator(std::vector<corr*> &corrs>){
  std::vector<corr*> condensed;
#endif

  condensed.resize(corrs.size());
  condensed[0] =std::move(corrs[0]);
  int J = 0;
  bool flag;
  for(size_t I=1;I<corrs.size(); I++){
    flag = true;
    for(int i =0; i<=J;i++){
      if(compare_corr(condensed[i],corrs[I])){
	flag = false;
	break;
      }
    }
    if(!flag) continue;
    J++;
    condensed[J] = std::move(corrs[I]);
    if(condensed[J].get() ==nullptr || corrs[I].get()!=nullptr){
      std::cerr << "what?" << std::endl;;
      std::cerr << "condesed["<< J << "] :" << condensed[J].get() << std::endl;
      std::cerr << "corrs:    " << corrs[I].get() << std::endl;
      exit(1);
    }
  }
  corrs.clear(); // release the memory hold in corrs and bring its size to 0
  corrs.reserve(J); // reserved enougth memory for J unique_ptr
  condensed.resize(J); // keep the first J element of condensed, releasing the memory of all the others
  for(auto& corr : condensed){
    corrs.push_back(corr->make_unique_ptr());
  }
}
