/*******************************************************************************
*
* File input_parser.cpp
*
* Copyright (C) 2025  D'Anna Antonino

* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************
*
* parsing of input files for meson.c, tm_meson.c in their latest version
* avaliable on Jul 2025. See README for details:
*
* input_parser.x <input_file1> ...
*
*******************************************************************************/

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <array>
#include <memory>
#include <algorithm>
#include <regex>
#include <time.h>
#include "../include/Gamma.h"
#include "../include/propagator.h"
#include "../include/correlator.h"
#include "../include/file_manip.h"

void parse_argv(const std::vector<std::string> &argv, std::string &output_dir, std::vector<std::string> &input_files){
  input_files.reserve(argv.size()-1);
  for(auto itr = std::next(argv.begin(),1); itr!=argv.end(); std::advance(itr,1)){
    if(itr->compare(0,2,"-o")==0){
      if(itr->length()==2){
        std::advance(itr,1);
        output_dir = *itr;
      }
      else {
        output_dir = itr->substr(2,std::string::npos);
      }
      if(output_dir.back()!='/')
        output_dir.push_back('/');
    }
    else{
      input_files.push_back(*itr);
    }
  }
  input_files.shrink_to_fit();
}

int main(int argc, char* argv[]){
  if(argc<2){
    std::cout << "Scalar mode not available." << std::endl;
    std::cout << "Try: input_parser.x <input_file1> <input_file2> ... <input_filen>" << std::endl;
    exit(1);
  }
  std::string output_dir{};
  std::vector<std::string> input_filenames{};
  parse_argv(std::vector<std::string>(argv,argv+argc),output_dir,input_filenames);
  std::ifstream input_file;
  std::ofstream output_file;
  std::ofstream condense_file;
  int ncorr, nprop;
  std::vector<propagator> propagators;
  std::vector<std::unique_ptr<corr>> correlators;

  for(std::string filename : input_filenames){
    input_file.open(filename);
    if(!input_file.is_open()){
      std::cerr << "Error: file "<< filename << " not found. Moving on..." << std::endl;
      continue;
    }
    {
      std::string aux{};
      if (output_dir.size() !=0){
        size_t idx = filename.find_last_of("/");
        aux = output_dir + filename.substr(idx+1,std::string::npos);
          }
      else {
        aux = filename;
      }
      size_t idx =aux.find_last_of(".");
      aux  = aux.substr(0,idx) + "_corr.txt";
      output_file.open(aux);
      if(!output_file.is_open()){
        std::cerr << "Error: unable to open output file " << aux <<" . Moving on..." << std::endl;
        input_file.close();
        continue;
      }
      aux = aux.substr(0,idx) +"_condese_corr.txt";
      condense_file.open(aux);
      if(!condense_file.is_open()){
        std::cerr << "Error: unable to open condense file "<< aux << ". No condense file will be produced"<< std::endl;
      }
    }
    std::vector<int> aux= {0,0};
    read_measurements(input_file,aux);
    nprop = aux[0];
    ncorr = aux[1];

    propagators.clear();
    propagators.resize(nprop);

    read_all_propagator(input_file, propagators, nprop);
    read_all_corr(input_file,correlators,propagators, ncorr);

    for(int i=0; i<ncorr; i++){
      output_file << "[Correlator "<< i << " ]"<< std::endl;
      correlators[i]->print(output_file);
      output_file <<  std::endl;
    }

    condense_correlator(correlators);
    for(auto& corr : correlators){
      corr->print(condense_file);
      condense_file <<  std::endl;
    }

    input_file.close();
    output_file.close();
    condense_file.close();
  }
}
