function apply_rw(data::Array{Float64}, W::Matrix{Float64}, vcfg::Union{Nothing, Vector{Int32}}=nothing; id::Union{String, Nothing}=nothing, fs::Bool=false)
    nc = size(W,2)
    if isnothing(vcfg)
        vcfg = collect(1:nc)
    end
    
    if fs == false
        if id == "A653"
            rw1 = W[1, 1:nc] 
            rw = rw1
            data_r = data .* rw[vcfg]
            return (data_r, rw)
        else
            rw1 = W[1, 1:nc] 
            rw2 = W[2, 1:nc]
            rw = rw1 .* rw2
            data_r = data .* rw[vcfg]
            return (data_r, rw)
        end
    else
        rw1 = W[1, 1:nc] 
        rw2 = W[2, 1:nc]
        rw_s = [1.0 for i in 1:nc]
        if id in ["H105r001", "H105r002", "H105r005", "J303", "J303r003"]
            rw_s[flag_s[id]] .= -1.0
        end
        rw = rw1 .* rw2 .* rw_s
        data_r = data .* rw[vcfg]
        return (data_r, rw)
    end
end

function apply_rw(data::Vector{<:Array{Float64}}, W::Vector{Matrix{Float64}}, vcfg::Union{Nothing, Vector{Vector{Int32}}}=nothing; id::Union{String, Nothing}=nothing, fs::Bool=false)
    nc = size.(W, 2)
    if isnothing(vcfg)
        vcfg = [collect(1:nc[k]) for k=1:length(nc)]
    end

    if fs == false
        if id == "A653"
            rw1 = [W[k][1, 1:nc[k]] for k=1:length(W)]    
            rw = [rw1[k] for k =1:length(W)]
            data_r = [data[k] .* rw[k][vcfg[k]] for k=1:length(data)]
            return (data_r, rw)
        else
            rw1 = [W[k][1, 1:nc[k]] for k=1:length(W)]
            rw2 = [W[k][2, 1:nc[k]] for k=1:length(W)]
            rw = [rw1[k] .* rw2[k] for k =1:length(W)]
            data_r = [data[k] .* rw[k][vcfg[k]] for k=1:length(data)]
            return (data_r, rw)
        end
    else
        rw1 = [W[k][1, 1:nc[k]] for k=1:length(W)]
        rw2 = [W[k][2, 1:nc[k]] for k=1:length(W)]
        rw_s = [[1.0 for i in 1:nc[k]] for k=1:length(W)]
        if id == "H105"
            rw_s[1][flag_s["H105r001"]] .= -1.0
            rw_s[2][flag_s["H105r002"]] .= -1.0
        elseif id == "H105r005"
            rw_s[1][flag_s["H105r005"]] .= -1.0
        elseif id == "J303" || id == "J303r003"
            rw_s[1][flag_s["J303r003"]] .= -1.0
        end
        rw = [rw1[k] .* rw2[k] .* rw_s[k] for k =1:length(W)]
        data_r = [data[k] .* rw[k][vcfg[k]] for k=1:length(data)]
        return (data_r, rw)
    end
end

function check_corr_der(obs::Corr, derm::Vector{Corr}; new_version::Bool=false)   
    g1 = Vector{String}(undef, 0)
    g2 = Vector{String}(undef, 0)

    for d in derm
        if !new_version
            aux = [d.gamma[1], d.gamma[2]]
            push!(g1, aux[1][1:end-3])
            push!(g2, aux[2][1:end-3])
        else
            aux = [d.gamma[1], d.gamma[2]]
            push!(g1, aux[1][1:end])
            push!(g2, aux[2][1:end])
        end
    end

    h = copy(derm)
    push!(h, obs)
    
    
    if any(getfield.(h, :y0) .!= getfield(h[1], :y0))
        return false
    end
    for s in [:kappa, :mu]
        for k = 1:2
            if any(getindex.(getfield.(h, s), k) .!= getindex(getfield(h[1], s), k))
                return false
            end
        end
    end
    #gamma check
    if any(g1 .!= obs.gamma[1]) || any(g2 .!= obs.gamma[2])
        return false
    end

    return true
end

@doc raw"""
    corr_obs(cdata::CData; real::Bool=true, rw::Union{Array{Float64, 2}, Nothing}=nothing, L::Int64=1, info::Bool=false, idm::Union{Vector{Int64},Nothing}=nothing, nms::Int64=Int64(maximum(cdata.vcfg)), flag_strange::Bool=false

    corr_obs(cdata::Array{CData, 1}; real::Bool=true, rw::Union{Array{Array{Float64, 2}, 1}, Nothing}=nothing, L::Int64=1, info::Bool=false, replica::Union{Vector{Int64},Nothing}=nothing, idm::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64, Nothing}=nothing, flag_strange::Bool=false)

Creates a `Corr` struct with the given `CData` struct `cdata` (See also [`read_mesons`](@ref)) for a single replica.
An array of `CData` can be passed as argument for multiple replicas.

# Arguments
  - `real::Bool = true`: if true select the real part of the correlator, otherwise select the immaginary one
  - `rw`: reweighting factor. When given, the method applies the reweighting. (See also [`read_ms1`](@ref)). `rw` has to be structered as and `Array{Float64,2}`, if the data have only one replica, or as `Array{Array{Float64,2}}`, if the data have more replicas
  - `L::Int64=1`: spatial size of the lattice. It is used to normalize the correlators with the volume `L^3`
  - `info::Bool =false`: if set to `true`, the method outputs the primary observable `<O>` if `rw=nothing` or `<OW>` and `<W>`  if `rw` has been given
  - `replica::Vector{Int64}`: length of the montecarlo replicas
  - `nms::Int64`: total length of the montecarlo data. Must be equal to `sum(replica)`
  - `idm::Vecto{Int64}`: configuration index where the measurements where taken. If multiple replicas are present, `idm` is still a single vector with the `idm` of each replica concatenated in replica-order
  - `flag_strange::Bool = false`: if set and rw is passed, it will reweight to undo negative sign of the strange quark determinant

# Example: 
 - Single replica 
```@example
data = read_mesons(path, "G5", "G5")
rw = read_ms1(path_rw)
corr_pp = corr_obs.(data)
corr_pp_r = corr_obs.(data, rw=rw)
```
 - Single replica + Info
```@example
data = read_mesons(path, "G5", "G5")
rw = read_ms1(path_rw)
corr_pp, O = corr_obs(data[1], info=true)
corr_pp_r, WO, W = corr_obs(data[1], rw=rw, info=true)
```
 - Two replicas
```@example
data = read_mesons([path_r1, path_r2], "G5", "G5")
rw1 = read_ms1(path_rw1)
rw2 = read_ms1(path_rw2)

corr_pp = corr_obs.(data)
corr_pp_r = corr_obs.(data, rw=[rw1, rw2])
```
 -  Replica with gaps in measurements. Let's say 2 replica with  montecarlo length [101,200], measurement taken only on even configuration
 
 ```@example
data = read_mesons([path_r1,path_r2]],"G5","G5")

corr_pp = corr_obs.(data,replica=[101,200],nms = 301, idm = [2:2:101...;2:2:200...]);
                                                      #      replica1   replica2    idm is a single vector with the idm of each replica concatenated 
```   
"""
function corr_obs(cdata::CData; real::Bool=true, rw::Union{Array{Float64, 2}, Nothing}=nothing, L::Int64=1, info::Bool=false, idm::Union{Vector{Int64},Nothing}=nothing, nms::Int64=Int64(maximum(cdata.vcfg)),replica = 0, flag_strange::Bool=false)

    real ? data = cdata.re_data ./ L^3 : data = cdata.im_data ./ L^3
    nt = size(data)[2]

    idm = isnothing(idm) ? Int64.(cdata.vcfg) : idm

    if isnothing(rw)
        # idm = isnothing(idm) ? collect(1:nms) : idm
        obs = [uwreal(data[:, x0], cdata.id, idm, nms) for x0 = 1:nt]
    else
        # idm = isnothing(idm) ? collect(1:nms) : idm
        data_r, W = apply_rw(data, rw, cdata.vcfg, id=cdata.id, fs=flag_strange)
        ow = [uwreal(data_r[:, x0], cdata.id, idm, nms) for x0 = 1:nt]
        W_obs = uwreal(W, cdata.id, idm, nms)
        obs = [ow[x0] / W_obs for x0 = 1:nt]
    end

    if info && !isnothing(rw)
        return (Corr(obs, cdata), ow, W_obs)
    elseif info && isnothing(rw)
        return (Corr(obs, cdata), obs)
    else
        return Corr(obs, cdata)
    end
end
#function corr_obs for R != 1
#TODO: vcfg with gaps
function corr_obs(cdata::Array{CData, 1}; real::Bool=true, rw::Union{Array{Array{Float64, 2}, 1}, Nothing}=nothing, L::Int64=1, info::Bool=false, replica::Union{Vector{Int64},Nothing}=nothing, idm::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64, Nothing}=nothing, flag_strange::Bool=false)
    nr = length(cdata)
    id = getfield.(cdata, :id)
    vcfg = getfield.(cdata, :vcfg)
    replica = isnothing(replica) ? Int64.(maximum.(vcfg)) : replica

    nms = isnothing(nms) ?  sum(replica) : nms

    nr = length(vcfg)
    if isnothing(idm)
        a = vcfg[1]
        for i in 2:nr
            a = [a; a[end] .+ vcfg[i]]
        end
        idm = Int64.(a)
    end
    
    if !all(id .== id[1])
        error("IDs are not equal")
    end

    data = real ? getfield.(cdata, :re_data) ./ L^3 :  getfield.(cdata, :im_data) ./ L^3

    nt = size(data[1])[2]

    if isnothing(rw)
        tmp = data[1]
        [tmp = cat(tmp, data[k], dims=1) for k = 2:nr]
        obs = [uwreal(tmp[:, x0], id[1], replica, idm, nms) for x0 = 1:nt]
    else
        data_r, W = apply_rw(data, rw, vcfg, id=id[1], fs=flag_strange)

        tmp = data_r[1]
        tmp_W = W[1]
        [tmp = cat(tmp, data_r[k], dims=1) for k = 2:nr]
        [tmp_W = cat(tmp_W, W[k], dims=1) for k = 2:nr]

        ow = [uwreal(tmp[:, x0], id[1], replica, idm, nms) for x0 = 1:nt]
        W_obs = uwreal(tmp_W, id[1], replica, idm, nms)
        obs = [ow[x0] / W_obs for x0 = 1:nt]
    end
    if info && !isnothing(rw)
        return (Corr(obs, cdata), ow, W_obs)
    elseif info && isnothing(rw)
        return (Corr(obs, cdata), obs)
    else
        return Corr(obs, cdata)
    end
end   

@doc raw"""
    corr_obs_TSM(cdata_sl::CData, cdata_corr::CData; real::Bool=true, rw::Union{Array{Float64, 2}, Nothing}=nothing, L::Int64=1, info::Bool=false, idm_sl::Union{Vector{Int64},Nothing}=nothing, idm_corr::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64,Nothing}=nothing, flag_strange::Bool=false)
    corr_obs_TSM(cdata1::Array{CData, 1}, cdata2::Array{CData, 1}; real::Bool=true, rw::Union{Array{Array{Float64, 2}, 1}, Nothing}=nothing, L::Int64=1, info::Bool=false, replica_sl::Union{Vector{Int64},Nothing}=nothing, idm_sl::Union{Vector{Int64},Nothing}=nothing, idm_corr::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64, Nothing}=nothing, flag_strange::Bool=false)
   

Creates a `Corr` struct  for Truncated Solver Method (TSM) with the given `CData` structs `cdata_sl` and `cdata_corr` for a single replica.
Two arrays of `CData` can be passed as argument for multiple replicas.

The flag `real` select the real or imaginary part of the correlator.
If `rw` is specified, the method applies reweighting. `rw` is passed as a matrix of Float64 (`read_ms1`)
The correlator can be normalized with the volume factor if `L` is fixed.

The flag `info` provides extra output that contains information about the primary observables. The function returns the primary observables ``<WO>`` and ``<W>``
(it returns the observable <O> if rw=nothing)

The flags `idm_sl`, `idm_corr` and `nms` can be used if the observable is not measure in every configuration. In particular:
`idm_sl::Vector{Int64}` If given, label the configurations where the sloppy observables is measured. Not required if sloppy is full statistics.    
`idm_corr::Vector{Int64}` If given, label the configurations where the correction observables is measured. Not required if correction is full statistics.
`nms::Int64` Total number of measurements in the ensembles. By default, nms=maximum(cdata_sl.vcfg) it is assumed that the sloppy observable is full statistics, while the exact observable is not.
Typically, only `idm_corr` has to be passed, if the exact observable is not full statistics. 
By playing with these three flags you ensure that the Gamma method is still working even if the configurations between sloppy and correction do not match.
`replica_sl::Vector{Int64}` allows to set the length of each replica independently of `maximum.(vcfg_sl)`. Be aware that `replica_sl` must be equal or larger than `maximum(vcfg_sl)`

```@example
#Single replica
data_sloppy      = read_mesons(path_sl, "G5", "G5")
data_correction  = read_mesons_correction(path_corr, "G5", "G5")

rw = read_ms1(path_rw)
corr_pp = corr_obs_TSM.(data_sloppy, data_correction)
corr_pp_r = corr_obs_TSM.(data_sloppy, data_correction rw=rw)

#Single replica + Info
data_sloppy      = read_mesons(path_sl, "G5", "G5")
data_correction  = read_mesons_correction(path_corr, "G5", "G5")
rw = read_ms1(path_rw)
corr_pp, O = corr_obs_TSM(data_sloppy, data_correction, info=true)
corr_pp_r, WO, W = corr_obs(data_sloppy, data_correction,, rw=rw, info=true)

#Two replicas
data = read_mesons([path_r1, path_r2], "G5", "G5")
rw1 = read_ms1(path_rw1)
rw2 = read_ms1(path_rw2)

corr_pp = corr_obs.(data)
corr_pp_r = corr_obs.(data, rw=[rw1, rw2])
```
"""
function corr_obs_TSM(cdata_sl::CData, cdata_corr::CData; real::Bool=true, rw::Union{Array{Float64, 2}, Nothing}=nothing, L::Int64=1, info::Bool=false, idm_sl::Union{Vector{Int64},Nothing}=nothing, idm_corr::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64,Nothing}=nothing, flag_strange::Bool=false)
    # cdata1 is sloppy, cdata2 is correction 
    if cdata_sl.id != cdata_corr.id
        error("Error: cdata_sl id != cdata_corr id")
    end
    if cdata_sl.header != cdata_corr.header # Base.:(==) and Base.:(!=) are redifined in juobs_types.jl
        error("Error: cdata_sl header != cdata_corr header")
    end

    id = getfield(cdata_sl, :id)

    vcfg_sl   = getfield(cdata_sl, :vcfg)
    vcfg_corr = getfield(cdata_corr, :vcfg)
    nms = isnothing(nms) ? Int64(maximum(vcfg_sl)) : nms # assuming vcfg_sl >= vcfg_corr
    
    idm_sl = isnothing(idm_sl) ? Int64.(vcfg_sl) : idm_sl
    idm_corr = isnothing(idm_corr) ? Int64.(vcfg_corr) : idm_corr
    

    data1 = real ? cdata_sl.re_data   ./ L^3 : cdata_sl.im_data   ./ L^3
    data2 = real ? cdata_corr.re_data ./ L^3 : cdata_corr.im_data ./ L^3

    nt = size(data1, 2) # 1
    # added automatic idsm detection  
    # nms_vec_sl   = cdata_sl.vcfg
    # nms_vec_corr = cdata_corr.vcfg
 
    # delta_cnfg_sl   = nms_vec_sl[2] - nms_vec_sl[1]
    # delta_cnfg_corr = nms_vec_corr[2] - nms_vec_corr[1]
 
    # idm_sl   = isnothing(idm_sl) ? collect(1:delta_cnfg_sl:nms_vec_sl[end]) : idm_sl
    # idm_corr = isnothing(idm_corr) ? collect(1:delta_cnfg_corr:nms_vec_corr[end]) : idm_corr
    # end section to be tested 
    if isnothing(rw)
        obs1 = [uwreal(data1[:, x0], id, idm_sl, nms) for x0 = 1:nt]
        obs2 = [uwreal(data2[:, x0], id, idm_corr, nms) for x0 = 1:nt]
    else
        data1_r, W_sl = apply_rw(data1, rw, vcfg_sl, id=id, fs=flag_strange)
        data2_r, W_corr = apply_rw(data2, rw, vcfg_corr, id=id, fs=flag_strange)
        ow1 = [uwreal(data1_r[:, x0], id, idm_sl, nms) for x0 = 1:nt]
        ow2 = [uwreal(data2_r[:, x0], id, idm_corr, nms) for x0 = 1:nt]

        W_obs_sl   = uwreal(W_sl, id, idm_sl, nms)
        W_obs_corr = uwreal(W_corr, id, idm_corr, nms)

        obs1 = [ow1[x0] / W_obs_sl for x0 = 1:nt]
        obs2 = [ow2[x0] / W_obs_corr for x0 = 1:nt]
    end

    if info && !isnothing(rw)
        return (Corr(obs1 + obs2, cdata_sl), ow1, ow2, W_obs_sl)
    elseif info && isnothing(rw)
        return (Corr(obs1 + obs2, cdata_sl), obs1, obs2)
    else
        return Corr(obs1 + obs2, cdata_sl)
    end

end
function corr_obs_TSM(cdata1::Array{CData, 1}, cdata2::Array{CData, 1}; real::Bool=true, rw::Union{Array{Array{Float64, 2}, 1}, Nothing}=nothing, L::Int64=1, info::Bool=false, replica_sl::Union{Vector{Int64},Nothing}=nothing, idm_sl::Union{Vector{Int64},Nothing}=nothing, idm_corr::Union{Vector{Int64},Nothing}=nothing, nms::Union{Int64, Nothing}=nothing, flag_strange::Bool=false)
    if any(getfield.(cdata1, :id) .!= getfield.(cdata2, :id))
        error("Error: cdata1 id != cdata2 id")
    end
    if any(getfield.(cdata1, :header) .!= getfield.(cdata2, :header)) # Base.:(==) and Base.:(!=) are redifined in juobs_types.jl
        error("Error: cdata1 header != cdata2 header")
    end
    
    id = getfield.(cdata1, :id)
    if !all(id .== id[1])
        error("IDs are not equal")
    end

    vcfg_sl      = getfield.(cdata1, :vcfg)
    replica_sl   = isnothing(replica_sl) ? Int64.(maximum.(vcfg_sl)) : replica_sl
    vcfg_corr    = getfield.(cdata2, :vcfg)
    replica_corr = Int64.(maximum.(vcfg_corr))
    
    nms = isnothing(nms) ? sum(replica_sl) : nms # assuming vcfg_sl >= vcfg_corr

    nr = length(vcfg_sl)
    if isnothing(idm_sl)
        a = vcfg_sl[1]
        for i in 2:nr
            a = [a; a[end] .+ vcfg_sl[i]]
        end
        idm_sl = Int64.(a)
    end
    if isnothing(idm_corr)
        a = vcfg_corr[1]
        for i in 2:nr
            a = [a; a[end] .+ vcfg_corr[i]]
        end
        idm_corr = Int64.(a)
    end

    data1 = real ? getfield.(cdata1, :re_data) ./ L^3 : getfield.(cdata1, :im_data) ./ L^3
    data2 = real ? getfield.(cdata2, :re_data) ./ L^3 : getfield.(cdata2, :im_data) ./ L^3

    nt = size(data1[1], 2)
    if isnothing(rw)
        tmp1 = cat(data1..., dims=1)
        tmp2 = cat(data2..., dims=1)

        obs1 = [uwreal(tmp1[:, x0], id[1], replica_sl, idm_sl, nms) for x0 = 1:nt]
        obs2 = [uwreal(tmp2[:, x0], id[1], replica_sl, idm_corr, nms) for x0 = 1:nt]

    else
        data1_r, W = apply_rw(data1, rw, vcfg_sl, id=id[1], fs=flag_strange)
        data2_r, W = apply_rw(data2, rw, vcfg_corr, id=id[1], fs=flag_strange)

        tmp1 = cat(data1_r..., dims=1)
        tmp2 = cat(data2_r..., dims=1)
        tmp_W = cat(W..., dims=1)

        ow1 = [uwreal(tmp1[:, x0], id[1], replica_sl, idm_sl, nms) for x0 = 1:nt]
        ow2 = [uwreal(tmp2[:, x0], id[1], replica_sl, idm_corr, nms) for x0 = 1:nt]

        W_obs = uwreal(tmp_W, id[1], replica_sl, idm_sl, nms)

        obs1 = [ow1[x0] / W_obs for x0 = 1:nt]
        obs2 = [ow2[x0] / W_obs for x0 = 1:nt]

    end
    if info && !isnothing(rw)
        return (Corr(obs1 + obs2, cdata1), ow1, ow2, W_obs)
    elseif info && isnothing(rw)
        return (Corr(obs1 + obs2, cdata1), obs1, obs2)
    else
        return Corr(obs1 + obs2, cdata1)
    end
end     
@doc raw"""
    corr_sym(corrL::Corr, corrR::Corr, parity::Int64=1)

Computes the symmetrized correlator using the left correlador `corrL` and the right correlator `corrR`. The source position
of `corrR` must be `T - 1 - y0`, where `y0` is the source position of `corrL`. 

```@example
pp_sym = corr_sym(ppL, ppR, +1)
a0p_sym = corr_sym(a0pL, a0pR, -1)
```
"""
function corr_sym(corrL::Corr, corrR::Corr, parity::Int64=1)
    T = length(corrL.obs)
    sym = [:kappa, :mu, :gamma]
    if corrL.y0 != T - 1 - corrR.y0
        error("Corr: Parameter mismatch")
    end
    for s in sym
        if getfield(corrL, s) != getfield(corrR, s)
            error("Corr: Parameter mismatch")
        end
    end
    if abs(parity) != 1
        error("incorrect value of parity (+- 1)")
    end

    res = (corrL.obs[1:end] + parity * corrR.obs[end:-1:1]) / 2
    return Corr(res, corrL.kappa, corrL.mu, corrL.gamma, corrL.y0, corrL.theta1, corrL.theta2)
end

function corr_sym_E250(corr1::Corr, corr2::Corr, parity::Int64=1)
    aux = [corr2.obs[97:end]; corr2.obs[1:96]]
    corr2_sym = Corr(aux, corr2.kappa, corr2.mu, corr2.gamma, corr1.y0, corr2.theta1, corr2.theta2)

    corr = [corr1.obs[1:3]; (corr1.obs[4:98] .+ parity * corr2_sym.obs[192:-1:98]) / 2]

    return Corr(corr, corr1.kappa, corr1.mu, corr1.gamma, corr1.y0, corr1.theta1, corr1.theta2)
end

function corr_sym_D450(corr1::Corr, corr2::Corr, parity::Int64=1)
    aux = [corr2.obs[65:end]; corr2.obs[1:64]]
    corr2_sym = Corr(aux, corr2.kappa, corr2.mu, corr2.gamma, corr1.y0, corr2.theta1, corr2.theta2)

    corr = [corr1.obs[1:3]; (corr1.obs[4:66] .+ parity * corr2_sym.obs[128:-1:66]) / 2]

    return Corr(corr, corr1.kappa, corr1.mu, corr1.gamma, corr1.y0, corr1.theta1, corr1.theta2)
end

#TODO: VECTORIZE, uwreal?
@doc raw"""
    md_sea(a::uwreal, md::Vector{Matrix{Float64}}, ow::uwreal, w::Union{uwreal, Nothing}=nothing, ws::ADerrors.wspace=ADerrors.wsg, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

    md_sea(a::uwreal, md::Vector{Matrix{Float64}}, ow::Array{uwreal}, w::Union{uwreal, Nothing}=nothing, ws::ADerrors.wspace=ADerrors.wsg, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

Computes the derivative of an observable A with respect to the sea quark masses.

``\frac{d <A>}{dm(sea)} = \sum_i \frac{\partial <A>}{\partial <O_i>}  \frac{d <O_i>}{d m(sea)}``


``\frac{d <O_i>}{dm(sea)} = <O_i> <\frac{\partial S}{\partial m}> - <O_i \frac{\partial S}{\partial m}>``

where ``O_i`` are primary observables. The function returns  ``\frac{\partial <A>}{\partial <O_i>}  \frac{d <O_i>}{d m(sea)}``, where
the primary observable ``O_i`` is specified as an input argument.

`ow` is a primary observable or a vector of primary observables that contains ``<OW>`` (``<O>`` if reweighting is not applied) and 
`w` is a primary obserable that contains the reweighting factors ``<W>`` (`w`=nothing if reweighting is not applied).

`md` is a vector that contains the derivative of the action ``S`` with respect
to the sea quark masses for each replica. `md[irep][irw, icfg]`

`md_sea` returns a tuple of uwreal observables ``(dA/dm_l, dA/dm_s)|_{sea}``, 
where ``m_l`` and ``m_s`` are the light and strange quark masses.

```@example
#Single replica
data = read_mesons(path, "G5", "G5")
md = read_md(path_md)
rw = read_ms1(path_rw)

corr_pp, wpp, w = corr_obs.(data[1], rw=rw, info=true)
m = meff(corr_pp[1], plat)
m_mdl, m_mds = md_sea(m, [md], wpp, w, ADerrors.wsg)
m_shifted = m + 2 * dml * m_mdl + dms * m_mds

#Two replicas
data = read_mesons([path_r1, path_r2], "G5", "G5")
md1 = read_md(path_md1)
md2 = read_md(path_md2)

corr_pp, pp = corr_obs(data[1], info=true)
m = meff(corr_pp[1], plat)
m_mdl, m_mds = md_sea(m, [md1, md2], pp, ADerrors.wsg)
m_shifted = m + 2 * dml * m_mdl + dms * m_mds
```
"""
function md_sea(a::uwreal, md::Vector{Matrix{Float64}}, ow::uwreal, w::Union{uwreal, Nothing}=nothing, ws::ADerrors.wspace=ADerrors.wsg, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)
    nid = neid(a)
    p = findall(t-> t==1, a.prop)

    if nid != 1
        @info("Error: neid > 1")
    end

    id = ws.map_nob[p]
    if !all(id .== id[1])
        @info("ids do not match")
    end
    id = ws.id2str[id[1]]
    
    ivrep = getfield.(ws.fluc[p], :ivrep)
    ivrep1 = fill(ivrep[1], length(ivrep))
    if !all(ivrep .== ivrep1)
        @info("ivreps do not match")
    end
    ivrep = ivrep[1]

    if length(md) != length(ivrep)
        @info("Nr obs != Nr md")
    end

    #md_aux as a Matrix + Automatic truncation
    md_aux = md[1][:, 1:ivrep[1]]
    for k = 2:length(md)
        md_aux = cat(md_aux, md[k][:, 1:ivrep[k]], dims=2)
    end
    
    nrw = size(md_aux, 1)
    # AD: We only need the mchist and mean value, we don't care about the error, so why we do need wpm? 
    # isnothing(wpm) ? uwerr(ow) : uwerr(ow,wpm) 
    uwerr(ow,Dict(id=>[1.0,-1.0,-1.0,-1.0]))
    ow_data = ow.mean .+ mchist(ow, id)

    if isnothing(w)
        der = derivative(a, ow)
        d1 = der != 0.0 ? der * (ow * uwreal(md_aux[1, :], id, ivrep) - uwreal(ow_data .* md_aux[1, :], id, ivrep)) : 0.0
        
        if nrw == 1
            return (d1, d1)
        elseif nrw == 2
            d2 = der != 0.0 ? der * (ow * uwreal(md_aux[2, :], id, ivrep) - uwreal(ow_data .* md_aux[2, :], id, ivrep)) : 0.0
            return (d1, d2)
        end
    else
        uwerr(w)
        der = derivative(a, ow) * w.mean
        w_data = w.mean .+ mchist(w, id)
        d1 = der != 0.0 ? der * (ow * uwreal(w_data .* md_aux[1, :], id, ivrep) / w^2 - uwreal(ow_data .* md_aux[1, :], id, ivrep) / w) : 0.0
        
        if nrw == 1
            return (d1, d1)
        elseif nrw == 2
            d2 = der != 0.0 ? der * (ow * uwreal(w_data .* md_aux[2, :], id, ivrep) / w^2 - uwreal(ow_data .* md_aux[2, :], id, ivrep) / w) : 0.0
            return (d1, d2)
        end
    end

    return nothing
end

function md_sea(a::uwreal, md::Vector{Matrix{Float64}}, ow::Array{uwreal}, w::Union{uwreal, Nothing}=nothing, ws::ADerrors.wspace=ADerrors.wsg, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)
    d = [md_sea(a, md, ow_, w, ws, wpm) for ow_ in ow]
    return (sum(getindex.(d, 1)), sum(getindex.(d, 2)))
end

@doc raw"""
    md_val(a::uwreal, obs::Corr, derm::Vector{Corr}; new_version::Bool=false, TSM::Bool=false)

Computes the derivative of an observable A with respect to the valence quark masses. 
If `new_version=true`, then the sign of the valence derivatives in `derm` is flipped 
(Julien's most updated code has the sign changed for the valence derivatives). Note 
also that for the new dat files version, no `_d1, _d2` is written by the reader, so 
you must select by hand (looking at the log files e.g.) which are you derivative 
correlators. In the log file, the derivative correlators are signaled by `seq_prop=some number`.

``\frac{d <A>}{dm(val)} = \sum_i \frac{\partial <A>}{\partial <O_i>}  \frac{d <O_i>}{d m(val)}``

``\frac{d <O_i>}{dm(val)} = <\frac{\partial O_i}{\partial m(val)}>``

where ``O_i`` are primary observables 

`md` is a vector that contains the derivative of the action ``S`` with respect
to the sea quark masses for each replica. `md[irep][irw, icfg]`

`md_val` returns a tuple of `uwreal` observables ``(dA/dm_1, dA/dm_2)|_{val}``, 
where ``m_1`` and ``m_2`` are the correlator masses.

```@example
data = read_mesons(path, "G5", "G5", legacy=true)
data_d1 = read_mesons(path, "G5_d1", "G5_d1", legacy=true)
data_d2 = read_mesons(path, "G5_d2", "G5_d2", legacy=true)

rw = read_ms1(path_rw)

corr_pp = corr_obs.(data, rw=rw)
corr_pp_d1 = corr_obs.(data_d1, rw=rw)
corr_pp_d2 = corr_obs.(data_d2, rw=rw)
derm = [[corr_pp_d1[k], corr_pp_d2[k]] for k = 1:length(pp_d1)]

m = meff(corr_pp[1], plat)
m_md1, m_md2 = md_val(m, corr_pp[1], derm[1])
m_shifted = m + dm1 * m_md1 + dm2 * m_md2
```
"""
function md_val(a::uwreal, obs::Corr, derm::Vector{Corr}; new_version::Bool=false)
    nid = neid(a)
    if nid != 1
        error("Error: neid > 1")
    end
    if length(derm) != 2
        error("Error: length derm != 2")
    end
    if !check_corr_der(obs, derm; new_version=new_version)
        error("Corr parameters does not match")
    end

    corr = getfield(obs, :obs)
    prop = getfield.(corr, :prop)

    if all(count.(prop) .== 1)
        der = [derivative(a, corr[k]) for k = 1:length(corr)]
    elseif all(count.(prop) .== 2)
        corr_der = getfield.(corr, :der)
        n = findall.(t-> t==1, prop)
        n = vcat(n'...)
        if all(n[:, 1] .== n[1, 1]) # find ow and w
            n_w = n[1, 1]
            n_ow = n[:, 2]
        else
            n_w = n[1, 2]
            n_ow = n[:, 1]
        end
        w_mean = 1 / getindex.(corr_der, n_ow)[1]
        #= Extra information (ow, w)
        ws = ADerrors.wsg
        ow_mean = - getindex.(corr_der, n_w) .* w_mean^2
        w_data = w_mean .+ ws.fluc[n_w].delta
        fluc_ow = getfield.(ws.fluc[n_ow], :delta)
        fluc_ow = vcat(fluc_ow'...)
        ow_data = ow_mean .+ ws.fluc[n_ow]
        =#
        der = length(a.der) >= maximum(n_ow) ? a.der[n_ow] * w_mean : 0.0
    else
        return nothing
    end
    derm1, derm2 = derm
    if !new_version
        return (sum(der .* derm1.obs), sum(der .* derm2.obs))
    else
        return (-sum(der .* derm1.obs), -sum(der .* derm2.obs))
    end
end

"""  
    plat_av(obs::Vector{uwreal}, plat::Vector{Int64}, [W::VecOrMat{Float64},] wpm::Union{Dict{Int64},Vector{Float{64}},Dict{String,Vector{Float{64}},Nothing}=nothing} )

It computes plateau average of `obs` in the range `plat`. Effectively this perform a fit to a constant. 
The field `W` is optional. When absent, `W` is a `Vector{Float64}` constructed from the error of `obs`. 
When `W` is a `Vector{Float64}` it performs an uncorrelated fit, when `W` is a `Matrix{Float64}`, it perfom a correlated fit

# Potential side effects

When `W` is absent, the function acts with `ADerror.uwerr` on `obs` in order to compute the errors.
This is the reason why a `wpm` may be given as input.

"""
function plat_av(obs::Vector{uwreal}, plat::Vector{Int64}, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)
    isnothing(wpm) ? uwerr.(obs) : [uwerr(obs_aux, wpm) for obs_aux in obs]
    w = 1 ./ err.(obs)[plat[1]:plat[2]].^2
    av = sum(w .* obs[plat[1]:plat[2]]) / sum(w)
    return av 
end

plat_av(obs::Vector{uwreal}, plat::Vector{Int64}, W::Vector{Float64},wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing) = sum(W .* obs[plat[1]:plat[2]]) / sum(W)




function model_av(fun::Function, y::Vector{uwreal}, guess::Float64; 
    tm::Vector{Int64}, tM::Vector{Int64}, k::Int64, 
    wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing) 
    
    pval = Array{Float64,1}()
    p_1 = Array{uwreal,1}()
    TIC = Array{Float64,1}()

    isnothing(wpm) ? uwerr.(y) : [uwerr(y[i], wpm) for i in 1:length(y)]
    for i in tm
        for j in tM
            if i < j
                x = collect(i:j)
                y_aux = y[i:j]
                try 
                    up, chi2, chi_exp, pval_i = fit_alg(fun,x,y_aux,k,guess,wpm=wpm)
                    push!(pval, pval_i)
                    push!(TIC, chi2 - 2*chi_exp)
                    push!(p_1, up[1])
                catch e 
                end
            end
        end
    end

    TIC = TIC .- minimum(TIC)
    weight = exp.(-0.5 .* TIC) ./ sum(exp.(-0.5 .* TIC))
    p_av = sum(p_1 .* weight)
    syst2 = sum(p_1 .^ 2 .* weight) - p_av ^ 2
    return p_av, syst2, p_1, weight, pval
end

function model_av(fun::Vector{Function}, y::Vector{uwreal}, guess::Float64; 
    tm::Vector{Vector{Int64}}, tM::Vector{Vector{Int64}}, k::Vector{Int64}, 
    wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing) 
    
    pval = Array{Float64,1}()
    p_1 = Array{uwreal,1}()
    TIC = Array{Float64,1}()

    isnothing(wpm) ? uwerr.(y) : [uwerr(y[i], wpm) for i in 1:length(y)]
    for ind in 1:length(fun)
        f = fun[ind]
        for i in tm[ind]
            for j in tM[ind]
                if i < j
                    x = collect(i:j)
                    y_aux = y[i:j]
                    try 
                        up, chi2, chi_exp, pval_i = fit_alg(f,x,y_aux,k[ind],guess,wpm=wpm)
                        push!(pval, pval_i)
                        push!(TIC, chi2 - 2*chi_exp)
                        push!(p_1, up[1])
                    catch e 
                    end
                end
            end
        end
    end

    TIC = TIC .- minimum(TIC)
    weight = exp.(-0.5 .* TIC) ./ sum(exp.(-0.5 .* TIC))
    p_av = sum(p_1 .* weight)
    syst2 = sum(p_1 .^ 2 .* weight) - p_av ^ 2
    return p_av, syst2, p_1, weight, pval
end

@doc raw"""
    bayesian_av(fun::Function, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k::Int64, pl::Bool, data::Bool; wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

    bayesian_av(fun1::Function, fun2::Function, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k1::Int64, k2::Int64, pl::Bool, data::Bool; wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

    bayesian_av(fun::Array{Function}, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k::Array{Int64}, pl::Bool, data::Bool; wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

Computes bayesian average of data. For a given fit function, it explores choices of fit intervals, assigning each of them a weight. The function saves the `first` fit parameter of your function, and then it does the weighted average of it and assigns a systematic. See https://arxiv.org/abs/2008.01069 

The function takes as input the fit intervals to explore. 

`tmin_array` is an array of integers with the lower bounds on the fit intervals to explore, ***ordered from lower to higher***.

`tmax_array` is an array of integers with the upper bounds on the fit intervals to explore, ***ordered from lower to higher***.

`k` is the number of parameters of the fit function to use.

You can also use as input two fit functions, and two values of `k`, one for each function. Then, for each fit interval choice, the function explores the two fit functions. This means that for each fit interval choice you get two results: one for the first fit funcction, and another for the second. You can also use a vector of functions and a vector of k (numer of parameters of each funtion) to apply the bayesian averaging method to multiple functions.

The method returns two objects: first, the weighted average as an uwreal object, with mean value and statistichal error. The second object returned is the systematic error coming from the fit interval variation. If `data` is `true`, then returns 4 objects: weighted average, systematic error, a vector with the results of the fit for each fit interval choice, and a vector with the weights associated to each fit.

```@example
@.fun(x,p) = p[1] * x ^0
k = 1
tmin_array = [10,11,12,13,14,15]
tmax_array = [80,81,82,83,84,85]
(average, systematics, data, weights) = bayesian_av(fun,x,tmin_array,tmax_array,k,pl=true,data=true)

@.fun1(x,p) = p[1] * x ^0
@.fun2(x,p) = p[1] + p[2] * exp( - p[3] * (x))
k1 = 1
k2 = 3
tmin_array = [10,11,12,13,14,15]
tmax_array = [80,81,82,83,84,85]
(average, systematics) = bayesian_av(fun1,fun2,x,tmin_array,tmax_array,k1,k2)

@.fun1(x,p) = p[1] * x ^0
@.fun2(x,p) = p[1] + p[2] * exp( - p[3] * (x))
k1 = 1
k2 = 3
tmin_array = [10,11,12,13,14,15]
tmax_array = [80,81,82,83,84,85]
(average, systematics) = bayesian_av([fun1,fun2],x,tmin_array,tmax_array,[k1,k2])
```
"""
function bayesian_av(fun::Function, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k::Int64; pl::Bool=false, data::Bool=false,
    wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing, path_plt::Union{String,Nothing}=nothing, 
    plt_title::Union{Nothing,String}=nothing, label::Union{Nothing, LaTeXString}=nothing)
    
    weight_model = Array{Float64,1}()
    AIC = Array{Float64,1}()
    chi2chi2exp = Array{Float64,1}()
    p1 = Array{uwreal,1}()
    mods = Array{String,1}()
    Pval = Array{Float64,1}()

    if tmax_array[end] > length(y)
        error("Error: upper bound for the fits is bigger than last data point")
    end

    total = length(y)
    isnothing(wpm) ? uwerr.(y) : [uwerr(y[i],wpm) for i in 1:length(y)]
    
    for INDEX in tmin_array ## vary tmin
        for j in tmax_array ## vary tmax
            try
                x = [i for i in INDEX+1:1:j]	
	        yy = y[INDEX+1:1:j]
	        Ncut = total - length(x)
                dy = err.(yy)
                W = 1 ./ dy .^2
                p00 = [0.5 for i in 1:1:k]
                chisq = gen_chisq(fun,x,dy)
                fit = curve_fit(fun,x,value.(yy),W,p00)
                isnothing(wpm) ? (up,chi_exp) = fit_error(chisq,coef(fit),yy) : (up,chi_exp) = fit_error(chisq,coef(fit),yy,wpm)
                isnothing(wpm) ? uwerr(up[1]) : uwerr(up[1],wpm)
                chi2 = sum(fit.resid.^2) # * dof(fit) / chi_exp
                Q = pvalue(chisq, sum(fit.resid.^2), value.(up), yy, wpm=wpm, W=W, nmc=10000)
                push!(Pval, Q)
                push!(AIC, chi2 - 2*chi_exp) # TIC criteria

                push!(chi2chi2exp, chi2 / dof(fit)) #substitute chi_exp with  dof(fit) for chi^2/dof
                push!(p1, up[1])
                push!(mods,string("[", INDEX+1, ",", j, "]"))
            catch
                @warn string(":/ Negative window for error propagation at tmin = ", INDEX, ", tmax = ", j, "; skipping that point")
            end
        end
    end
    
    # compute all weights
    offset = minimum(AIC)
    AIC = AIC .- offset
    weight_model = exp.(-0.5 .* AIC)
    weight_model = weight_model ./ sum(weight_model)
    
    # sort weights and discard 5% tail 
    idxW = sortperm(weight_model, rev=true)
    cumulative_w = cumsum(weight_model[idxW])
    # println("acceptance in bayeasian_av set to 0.99, should be 0.95")
    idxcumw = findfirst(x->x>=0.95, cumulative_w)
    # println(length(idxW) - idxcumw)
    idxW = sort(idxW[1:idxcumw])
    
    # take only the accepted 
    AIC = AIC[idxW]
    Pval = Pval[idxW]
    chi2chi2exp  = chi2chi2exp[idxW]
    mods = mods[idxW]
    p1 = p1[idxW]
    weight_model = weight_model[idxW]
    weight_model ./= sum(weight_model)


    
    p1_mean = sum(p1 .* weight_model)
    isnothing(wpm) ? uwerr(p1_mean) : uwerr(p1_mean, wpm) 
    # println(sum(p1 .^ 2 .* weight_model))
    # println((sum(p1 .* weight_model)) ^ 2)
    systematic_err =  0.0 #sqrt(sum(p1 .^ 2 .* weight_model) - (sum(p1 .* weight_model)) ^ 2) 
    if pl   
        fig = figure(figsize=(10,7.5))

        subplots_adjust(hspace=0.1) 
        subplot(411)    
        if !isnothing(plt_title)
            #title(plt_title)
        end
        ax1 = gca()                
        x = 1:length(p1)
        y = value.(p1)
        dy = err.(p1)
        v = value(p1_mean)
        e = err(p1_mean)

        fill_between(x, v-e, v+e, color="royalblue", alpha=0.2)
        errorbar(x, y, dy, fmt="^", mfc="none", color="navy", capsize=2)
        setp(ax1.get_xticklabels(),visible=false) # Disable x tick labels
        isnothing(label) ?  ylabel(L"$p_1$") : ylabel(label)
    
        subplot(412)
        ax2=gca()
        bar(x, weight_model, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5)
        setp(ax2.get_xticklabels(),visible=false) # Disable x tick labels
        # errorbar(mods, weight_model, 0*dy, color="green")
        ylabel(L"$W$")

        subplot(413)
        ax3=gca()
        setp(ax3.get_xticklabels(),visible=false) # Disable x tick labels

        bar(x, Pval, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5 )
        ylabel(L"$p-value$")
        
        
        subplot(414)
        bar(x, chi2chi2exp, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5 )
        # ylabel(L"$\chi^2/\chi^2_{\mathrm{exp}}$")
        ylabel(L"$\chi^2/\mathrm{d.o.f.}$")
        xticks(x, mods, rotation=45)
        xlabel(L"$\mathrm{Models} \ [t_{\mathrm{min}}/a,t_{\mathrm{max}}/a]$")
        tight_layout()
        display(fig)
        if !isnothing(path_plt)
            #tt = plt_title *".pdf" 
            savefig(path_plt)
        end
        close()
    end

    if !data                   
        return (p1_mean, systematic_err)                                         
    else               
        FitStorage = Dict(
        "W"        => weight_model,
        "AIC"      => AIC .+ offset,
        "pval"     => Pval, 
        "chivsexp" => chi2chi2exp,
        "mods"     => mods
        )
        return (p1_mean, systematic_err, p1, FitStorage)    
    end 

end

function bayesian_av(fun1::Function, fun2::Function, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k1::Int64, k2::Int64; pl::Bool=false, data::Bool=false, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)

    weight_model = Array{Float64,1}()
    AIC = Array{Float64,1}()
    chi2chi2exp = Array{Float64,1}()
    p1 = Array{uwreal,1}()
    mods = Array{String,1}()

    if tmax_array[end] > length(y)
        error("Error: upper bound for the fits is bigger than last data point")
    end

    total = length(y)
    isnothing(wpm) ? uwerr.(y) : for i in 1:length(y) uwerr(y[i],wpm) end
    
    for INDEX in tmin_array ## vary tmin
        for j in tmax_array ## vary tmax
            try
                x = [i for i in INDEX+1:1:j]	
	            yy = y[INDEX+1:1:j]
	            Ncut = total - length(x)
                dy = err.(yy)
                W = 1 ./ dy .^2

                p00 = [0.5 for i in 1:1:k1]
                chisq = gen_chisq(fun1,x,dy)
                fit = curve_fit(fun1,x,value.(yy),W,p00)
                isnothing(wpm) ? (up,chi_exp) = fit_error(chisq,coef(fit),yy) : (up,chi_exp) = fit_error(chisq,coef(fit),yy,wpm)
                isnothing(wpm) ? uwerr(up[1]) : uwerr(up[1],wpm)
                chi2 = sum(fit.resid.^2) * dof(fit) / chi_exp
                push!(AIC, chi2 + 2*k1 + 2*Ncut)
                push!(chi2chi2exp, chi2 / dof(fit))
                push!(p1, up[1])

                push!(mods,string("[", INDEX+1, ",", j, "]"))

                p00 = [0.5 for i in 1:1:k2]
                chisq = gen_chisq(fun2,x,dy)
                fit = curve_fit(fun2,x,value.(yy),W,p00)
                isnothing(wpm) ? (up,chi_exp) = fit_error(chisq,coef(fit),yy) : (up,chi_exp) = fit_error(chisq,coef(fit),yy,wpm)
                uwerr(up[1],wpm)
                chi2 = sum(fit.resid.^2) * dof(fit) / chi_exp
                push!(AIC, chi2 + 2*k2 + 2*Ncut)
                push!(chi2chi2exp, chi2 / dof(fit))
                push!(p1, up[1])
                println(up[end])
                push!(mods,string("[", INDEX+1, ",", j, "]"))

            catch e
                @warn string(":/ Negative window for error propagation at tmin = ", INDEX, ", tmax = ", j, "; skipping that point")
            end
        end
    end

    offset = minimum(AIC)
    AIC = AIC .- offset
    weight_model = exp.(-0.5 .* AIC)
    
    p1_mean = sum(p1 .* weight_model)/sum(weight_model) ; isnothing(wpm) ? uwerr(p1_mean) : uwerr(p1_mean,wpm) 
    weight_model = weight_model ./ sum(weight_model)
    systematic_err = sqrt(sum(p1 .^ 2 .* weight_model) - (sum(p1 .* weight_model)) ^ 2) 

    if pl                       
        x = 1:length(p1)
        y = value.(p1)
        dy = err.(p1)
        v = value(p1_mean)
        e = err(p1_mean)
        
        figure()
        fill_between(1:length(p1), v-e, v+e, color="green", alpha=0.75)
        errorbar(mods, y, dy, fmt="x", color="black")
        ylabel(L"$p_1$")
        xlabel(L"model")
    
        display(gcf())

        figure()
        errorbar(mods, weight_model, 0*dy, color="green")
        ylabel(L"$weight$")
        xlabel(L"model")
    
        display(gcf())
    end

    if !data                   
        return (p1_mean, systematic_err)                                         
    else               
        return (p1_mean, systematic_err, p1, weight_model)    
    end 

end

function bayesian_av(fun::Array{Function}, y::Array{uwreal}, tmin_array::Array{Int64}, tmax_array::Array{Int64}, k::Array{Int64}; pl::Bool=false,
    data::Bool=false, wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing, 
    path_plt::Union{String,Nothing}=nothing, plt_title::Union{Nothing,String}=nothing)

    weight_model = Array{Float64,1}()
    AIC          = Array{Float64,1}()
    chi2chi2exp  = Array{Float64,1}()
    p1           = Array{uwreal,1}()
    mods         = Array{String,1}()
    Pval         = Array{Float64,1}()

    if tmax_array[end] > length(y)
        error("Error: upper bound for the fits is bigger than last data point")
    end

    total = length(y)
    isnothing(wpm) ? uwerr.(y) : [uwerr(y[i],wpm) for i in 1:length(y)] 
    
    for INDEX in tmin_array ## vary tmin
        for j in tmax_array ## vary tmax
            try
                x = [i for i in INDEX+1:1:j]	
	            yy = y[INDEX+1:1:j]
	            Ncut = total - length(x)
                dy = err.(yy)
                W = 1 ./ dy .^2

                for indice in 1:length(fun)
                    p00 = [0.5 for i in 1:1:k[indice]]
                    chisq = gen_chisq(fun[indice],x,dy)
                    fit = curve_fit(fun[indice],x,value.(yy),W,p00)
                    isnothing(wpm) ? (up,chi_exp) = fit_error(chisq,coef(fit),yy) : (up,chi_exp) = fit_error(chisq,coef(fit),yy,wpm)
                    isnothing(wpm) ? uwerr(up[1]) : uwerr(up[1],wpm)
                    chi2 = sum(fit.resid.^2) * dof(fit) / chi_exp
                    Q = pvalue(chisq, sum(fit.resid.^2), value.(up), yy, wpm=wpm, W=W, nmc=10000)
                    if chi2 /  dof(fit) < 5.
                        # if err(up[1])/value(up[1])*100 >= 10.
                        # error()
                        # end
                        push!(Pval, Q)
                        push!(AIC, chi2 + 2*k[indice] + 2*Ncut)
                        push!(chi2chi2exp, chi2 / dof(fit))
                        push!(p1, up[1])
                        push!(mods,string("[", INDEX+1, ",", j, "]"))
                    end
                end

            catch e
                @warn string(":/ Negative window for error propagation at tmin = ", INDEX, ", tmax = ", j, "; skipping that point")
            end
        end
    end
    # compute all weights
    offset = minimum(AIC)
    AIC = AIC .- offset
    weight_model = exp.(-0.5 .* AIC)
    weight_model = weight_model ./ sum(weight_model)
    
    # sort weights and discard 5% tail 
    idxW = sortperm(weight_model, rev=true)
    cumulative_w = cumsum(weight_model[idxW])
    idxcumw = findfirst(x->x>=0.95, cumulative_w)
    println(length(idxW) - idxcumw)
    idxW = sort(idxW[1:idxcumw])
    
    # take only the accepted 
    AIC = AIC[idxW]
    Pval = Pval[idxW]
    chi2chi2exp  = chi2chi2exp[idxW]
    mods = mods[idxW]
    p1 = p1[idxW]
    weight_model = weight_model[idxW]
    weight_model /= sum(weight_model)
    
    p1_mean = sum(p1 .* weight_model) 
    isnothing(wpm) ? uwerr(p1_mean) : uwerr(p1_mean,wpm) 
    systematic_err = sqrt(sum(p1 .^ 2 .* weight_model) - (sum(p1 .* weight_model)) ^ 2) 

    if pl   
        subplots_adjust(hspace=0.1) 
        subplot(411)    
        if !isnothing(plt_title)
            title(plt_title)
        end
        ax1 = gca()                
        x = 1:length(p1)
        y = value.(p1)
        dy = err.(p1)
        v = value(p1_mean)
        e = err(p1_mean)

        fill_between(x, v-e, v+e, color="royalblue", alpha=0.2)
        errorbar(x, y, dy, fmt="^", mfc="none", color="navy", capsize=2)
        setp(ax1.get_xticklabels(),visible=false) # Disable x tick labels
        ylabel(L"$p_1$")
    
        subplot(412)
        ax2=gca()
        bar(x, weight_model, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5)
        setp(ax2.get_xticklabels(),visible=false) # Disable x tick labels
        # errorbar(mods, weight_model, 0*dy, color="green")
        ylabel(L"$W$")

        subplot(413)
        ax3=gca()
        setp(ax3.get_xticklabels(),visible=false) # Disable x tick labels

        bar(x, Pval, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5 )
        ylabel(L"$p\ value$")
        
        
        subplot(414)
        bar(x, chi2chi2exp, alpha=0.4, color="forestgreen", edgecolor="darkgreen", linewidth=1.5 )
        ylabel(L"$\chi^2/\chi^2_{\mathrm{exp}}$")
        xticks(x, mods, rotation=45)
        xlabel(L"$Models$")

        display(gcf())
        if !isnothing(path_plt)
            tt = plt_title *".pdf" 
            savefig(joinpath(path_plt, tt))
        end
        close()
    end
    

    if !data                   
        return (p1_mean, systematic_err)                                         
    else               
        FitStorage = Dict(
        "W"        => weight_model,
        "AIC"      => AIC .+ offset,
        "pval"     => Pval, 
        "chivsexp" => chi2chi2exp,
        "mods"     => mods
        )
        return (p1_mean, systematic_err, p1, FitStorage)
    end 

end

function lin_fit(x::Vector{<:Real}, v::Vector{Float64}, e::Vector{Float64})
    sig2 = e .* e
    S = sum(1 ./ sig2)
    Sx = sum(x ./ sig2)
    Sy = sum(v ./ sig2)
    Sxy = sum(v .* x ./ sig2)
    Sxx = sum(x .* x ./sig2)
    delta = S * Sxx - Sx*Sx
    par = [Sxx*Sy-Sx*Sxy, S*Sxy-Sx*Sy] ./delta
    #C = [[Sxx/delta, -Sx/delta], [-Sx/delta,  S/delta]]
    return par
end
@doc raw"""
    lin_fit(x::Vector{<:Real}, y::Vector{uwreal})

Computes a linear fit of uwreal data points y. This method return uwreal fit parameters and chisqexpected.

```@example
fitp, csqexp = lin_fit(phi2, m2)
m2_phys = fitp[1] + fitp[2] * phi2_phys
```
"""
function lin_fit(x::Vector{<:Real}, y::Vector{uwreal}; wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}, Nothing}=nothing)
    isnothing(wpm) ? uwerr.(y) : [uwerr(yaux, wpm) for yaux in y]
    par = lin_fit(x, value.(y), err.(y))
    chisq(p, d) = sum((d .- p[1] .- p[2].*x).^2 ./ err.(y) .^2)
    (fitp, csqexp) = fit_error(chisq, par, y)
    for i = 1:length(fitp)
        isnothing(wpm) ? uwerr(fitp[i]) : uwerr(fitp[i], wpm)
        print("\n Fit parameter: ", i, ": ")
        details(fitp[i])
    end
    println("Chisq / chiexp: ", chisq(par, y), " / ", csqexp, " (dof: ", length(x)-length(par),")")
    return (fitp, csqexp)
end

@doc raw"""
    x_lin_fit(par::Vector{uwreal}, y::Union{uwreal, Float64})

Computes the results of a linear interpolation/extrapolation in the x axis
"""
x_lin_fit(par::Vector{uwreal}, y::Union{uwreal, Float64}) = (y - par[1]) / par[2]
@doc raw"""
    y_lin_fit(par::Vector{uwreal}, y::Union{uwreal, Float64})

Computes the results of a linear interpolation/extrapolation in the y axis
"""
y_lin_fit(par::Vector{uwreal}, x::Union{uwreal, Float64}) = par[1] + par[2] * x


@doc """
    get_chi(model::Function,x::Vector,par,data,W::VecOrMat{Float64})

It computes the chi square of a fit. `model` is assumed to be of the type f(xdata,par). 
If `W::Vector` the chisquare is computed for an uncorrelated fit
If `W::Matrix` the chisquare is computed for a correlated fit
"""
function get_chi(model::Function,x,par,data,W::Array{Float64,2})
  return sum((data.-model(x,par))' * W * (data.-model(x,par)))
end

function get_chi(model::Function,x,par,data,W::Vector{Float64})
  return sum((data.-model(x,par)).^2 .* W)
end

@doc raw"""
        fit_routine(model::Function,xdata::AbstractArray{<:Real},ydata::AbstractArray{ADerrors.uwreal},npar; kwargs...)
        fit_routine(model::Function,xdata::AbstractArray{uwreal},ydata::AbstractArray{ADerrors.uwreal},npar; kwargs...)

It executes a fit of the `ydata` with the fit function `model`. 

# Arguments:

  - `model`: Fit function used to fit the data. the function assumes that its called as `model(xdata,parameters)`.
  - `xdata`: indipendent variable in the fit. It can be an `AbstractArray{<:Real}`, that is the "default" method, or 
             a `AbstractArray{uwreal}`. In the latter case, the method builds the function
``math
F(q_i) = \begin{cases}
         model(q_{npar+i},q[1:npar])  \quad \text{if} i<= length(xdata)\\
         q_i \quad \text{elsewhere}
       \end{cases}
``
             and then call the default method with dummy `xdata` and `vcat(ydata,xdata)` as new `ydata`
  - `ydata`: Data variable that carries the error 
  - `npar`: number of parameters used in the fit.

# Keyword parameters:

  - `wpm`: window parameter used by ADerrors to computes the errors. Default to `Dict{Int64,Vector{Float64}}()`.
           This is used both when computing the error of the data and when computing the covariance matrix through `ADerrors.cov`
           It accept a dictionary with ensemble id as keys (either `String` or `Int64`) and `Vector{Float64}` as value.
           See also ADerrors documentation.

  - `W::AbstractVecOrMat{Float64}`: Weight matrix/vector. If `W` is a `Vector` the fit is perfomed uncorrelated ,otherwise is correlated.
                                    default to `Vector{Float64}()`

  - `corr::Bool`: flag used to generated `W` when not given. Default to `true`. If `false`, `W = [1/y.err^2 for y in ydata]`.
                  If `true` the covariance matrix of the data is used to compute `W` as `W = C^{-1}`.

  - `C::AbstractMatrix`: covariance matrix used to compute W. Default to `Matrix{Float64}(undef, 0, 0)`.
                         If `W` is not given, and `corr` is `true`, then it is used to compute `W`.
                         If need to computed `W` but not given,  then `C = ADerrors.cov(ydata,wpm)`. See also ADerrors documentation.
                         If available, it will be used to better estimate the fit pvalue and the expected chi-square,
                         but it will not be computed if not available at this point. 

  - `guess::Vector{Float64}`: Initial guess for the parameter used in the fit routine. If not given, default to `fill(0.5,npar)`.
                              If `xdata isa Vector{uwreal}`, the mean value of `xdata` are used as guess for the relevant `q_i` parameters.
  
  - `logfile`: handle to a logfile. If `nothing`, no log will be written. It can be `IO` file or a custom log structure that has
               an overloading for `print()` and `println()`.

# Returns
It returns a `NamedTuple` with names:
  - `:par`:    fit parameter as `Vector{uwreal}`
  - `:chi2`:   chisquare,
  - `:chiexp`: chisquare expected 
  - `:pval`:   pvalue 
"""
function fit_routine(model::Function,
                     xdata::AbstractArray{<:Real},
                     ydata::AbstractArray{uwreal},
                     npar::Int64; 
                     wpm::Union{Dict{Int64,Vector{Float64}},Dict{String,Vector{Float64}}}=Dict{Int64,Vector{Float64}}(),
                     corr::Bool = true, 
                     W::AbstractArray{Float64}=Vector{Float64}(),
                     guess::AbstractVector{Float64} = fill(0.5,npar),
                     logfile = nothing,
                     C::AbstractArray{Float64} = Matrix{Float64}(undef,0,0));

    nobs = length(ydata)
    if length(xdata)!=nobs
        error("xdata and ydata must be the of same size")
    end

    [uwerr(y, wpm) for y in ydata]
    if length(W) ==0
        if corr 
            C = length(C) ==0 ? GammaMethod.cov_AD(ydata,wpm) : C
            W = LinearAlgebra.pinv(C);
            W = (W'+W)*0.5
        else
            W = [1/y.err^2 for y in ydata]
        end
    end

    fit = LsqFit.curve_fit(model,xdata,ADerrors.value.(ydata),W,guess)

    chisq(p,d) = get_chi(model,xdata,p,d,W)
    chi2 = sum(fit.resid.^2)
    par = fit_error(chisq,LsqFit.coef(fit),ydata,wpm,W,false)
    # if the fit is correlated  we have already C, no need to recompute it.
    chiexp,pval =  if length(C) !=0 
        GammaMethod.chiexp(chisq,LsqFit.coef(fit),value.(ydata),C,W),juobs.pvalue(chisq,chi2,LsqFit.coef(fit),ydata,W,C=C)
    else  
        ADerrors.chiexp(chisq,LsqFit.coef(fit),ydata,wpm,W=W),juobs.pvalue(chisq,chi2,LsqFit.coef(fit),ydata,W)
    end

    if !isnothing(logfile)
        uwerr.(par);
        println(logfile, "juobs.fit_routine output:")
        println(logfile, "\t\tFit routine in [$(xdata[1]),...,$(xdata[end])]")
        println(logfile, "\t\tparameters :")
        for p in par
            println(logfile,"\t\t\t",p)
        end
        println(logfile, "\t\tchi2:       ", chi2)
        println(logfile, "\t\tchiexp:     ", chiexp)
        println(logfile, "\t\tpvalue:     ", pval)
    end

    return (par=par,chi2=chi2,chiexp=chiexp,pval=pval)
end

"""
        check_size(M,n)

Check if `M` has sizes all equal to n or if is empty
"""
function check_sizes(M,n)
    if length(M) ==0 || all(size(M) .== n)
        return true
    end
    return false
end

function fit_routine(model::Function,
                     xdata::AbstractArray{ADerrors.uwreal},
                     ydata::AbstractArray{ADerrors.uwreal},
                     npar::Int64;
                     W::AbstractArray = Float64[],
                     wpm::AbstractDict = Dict{Int64,Vector{Float64}}(),
                     corr::Bool = true,
                     guess::AbstractArray = fill(0.5,npar),
                     logfile = nothing,
                     C::AbstractArray = Float64[])
    Nx = length(xdata)
    if (Nx != length(ydata))
        error("[juobs] Error: xdata and ydata must have the same dimension")
    end
    Ndata = 2Nx; ## in total we have 2*lengt(xdata) data points
    data = vcat(ydata,xdata)
    if !check_sizes(W,Ndata)
        error("[juobs] Error: W does not have the correct sizes")
    end
    if !check_sizes(C,Ndata)
        error("[juobs] Error: C does not have the correct sizes")
    end

    if corr
        C = length(C) ==0 ? GammaMethod.cov_AD(data) : C
        W = length(W) ==0 ? LinearAlgebra.pinv(W) : W
    else
        W = length(W) ==0 ? [1/yy.err^2 for yy in data] : W
    end

    f(q,p) = [i<=Nx ? model(q[i],p) : q[i-Nx] for i in 1:Ndata]
    fit_func(x,p) = f(view(p,npar+1:npar+Nx),view(p,1:npar))
    new_guess = [guess; value.(xdata)]
    fit = fit_routine(fit_func,zeros(Ndata),data,npar+Nx,
                      guess=new_guess, wpm=wpm,corr=corr,
                      W=W,C=C,logfile=nothing)

    if !isnothing(logfile) 
        uwerr.(fit.par)
        println(logfile, "juobs.fit_routine output:")
        println(logfile, "\t\txdata carries uncertainty")
        println(logfile, "\t\tFit routine in [$(xdata[1].mean),...,$(xdata[end].mean)]")
        println(logfile, "\t\tFit parameters:")
        for i in 1:npar
            println(logfile,"\t\t\t",fit.par[i])
        end
        println(logfile, "\t\tauxiliary fit parameter:")
        println(logfile, "\t\t\txdata[i] \t q[i]")
        for i in 1:Nx
            println(logfile, "\t\t\t",xdata[i]," \t ", fit.par[npar+i])
        end
        println(logfile, "\t\tchi2:       ", fit.chi2)
        println(logfile, "\t\tchiexp:     ", fit.chiexp)
        println(logfile, "\t\tpvalue:     ", fit.pval)
    end
    return fit
end


@doc raw"""
        fit_routine(model::AbstractArray{Function},
                    xdata::AbstractArray{<:AbstractArray},
                    ydata::AbstractArray{<:AbstractArray{ADerrors.uwreal}},npar; kwargs...)
       

It executes a combined fit of the `ydata` with the fit functions in `model`. The method define the general model `F(X_{mi},p) = model[m](xdata[i],p)`
and call `fit_routine(F,vcat(xdata),vcat(ydata),npar;kwargs....)`. The `kwargs` parameters are accordingly updated to reflect the new function.

# Returns
It returns a `NamedTuple` with names:
  - `:par`:    fit parameter as `Vector{uwreal}`
  - `:chi2`:   chisquare,
  - `:chiexp`: chisquare expected 
  - `:pval`:   pvalue 
"""
function fit_routine(models::AbstractArray{Function},
                     xdata::AbstractArray{<:AbstractArray},
                     ydata::AbstractArray{<:AbstractArray{ADerrors.uwreal}},
                     npar::Int64;
                     W::AbstractArray = Float64[],
                     C::AbstractArray = Float64[],
                     wpm::AbstractDict = Dict{Int64,Vector{Float64}}(),
                     corr::Bool = true,
                     guess::AbstractArray{<:AbstractArray} = [fill(0.5,npar[i]) for i in eachindex(models)],
                     logfile = nothing,
                     )
    Nmodel = length(models)
    if Nmodel != length(xdata) != length(ydata)
        error("[juobs] Error: you need the same number of model, xdata and ydata")
    end
    if !(length(W) == 0 || length(W)==Nmodel)
        error("[juobs] Error: You need to pass a W matrix for model or none")
    end
    if !(length(C) == 0 || length(C)==Nmodel)
        error("[juobs] Error: You need to pass a C matrix for model or none")
    end
    if length(guess)!=Nmodel
        error("[juobs] Error: You need to specify the inital guess for all model or for none")
    end
    ndata = length.(xdata)
    if !all(length.(ydata).== ndata)
        error("[juobs] Error: Mismatch in xdata and ydata. Make sure that the data corresponds")
    end
    Ntotal = sum(ndata);
    X = vcat(xdata...)
    Y = vcat(ydata...)

    function make_matrix(M)
        aux = zeros(Ntotal,Ntotal)
        ie = 0
        for m in 1:Nmodel
            is = ie+1
            ie += ndata[m];
            aux[is:ie,is:ie] .= M[m]
        end
        return aux
    end

    WW = length(W)==0 ? W : make_matrix(W);
    CC = length(C)==0 ? C : make_matrix(C);

    function Model(x,p)
        res = zeros(Nmodel)
        res[1] = models[1](x[1:ndata[1]],p)
        for i in 2:Nmodel
            rd = (ndata[i-1]+1):ndata[i]
            rp = (npar[i-1]+1):npar[i]
            res[i] = models[i](x[rd],p)
        end
        return res
    end
    Guess = vcat(guess...)
    fit = fit_routine(Model,X,Y,npar,guess = Guess,W=WW,C=CC,corr=corr,logfile=nothing,wpm=wpm)

    if !isnothing(logfile) 
        uwerr.(fit.par)
        flag = typeof(xdata) <: AbstractArray{<:AbstractArray{ADerrors.uwreal}}
        println(logfile, "juobs.fit_routine output:")
        if flag
            println(logfile, "\t\txdata carries uncertainty")
        end
        println(logfile, "\t\tGlobal fit with $Nmodel functions")
        for nm in 1:Nmodel
            println(logfile,"---Model $(nm)---")
            println(logfile, "\t\tFit routine in [$(xdata[nm][1].mean),...,$(xdata[nm][end].mean)]")
        end
        println(logfile, "\t\tFit parameters:")
        for i in 1:npar
            println(logfile,"\t\t\t",fit.par[i])
        end
        if flag
            off = npar
            pritnln(logfile, "\t\tauxiliary fit parameter:")
            for nm in 1:Nmodel
                println(logfile,"---Model $(nm)---")
                println(logfile, "\t\t\txdata[m][i] \t q[i]")
                for i in eachindex(xdata[nm])
                    println(logfile, "\t\t\t",xdata[nm][i]," \t ", fit.par[off+i])
                end
                off += npar+length(xdata[nm])
            end
        end
        println(logfile, "\t\tchi2:       ", chi2)
        println(logfile, "\t\tchiexp:     ", chiexp)
        println(logfile, "\t\tpvalue:     ", pval)
    end
    return fit
end

function fve(mpi::uwreal, mk::uwreal, fpi::uwreal, fk::uwreal, ens::EnsInfo)
    mm = [6,12,8,6,24,24,0,12,30,24,24,8,24,48,0,6,48,36,24,24]
	nn = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20]

    meta = sqrt(4/3 * mk ^ 2 - 1/3 * mpi ^ 2)

    jipi = value(mpi) ^ 2 / (4*pi*fpi) ^ 2
    jik = value(mk) ^ 2 / (4*pi*fpi) ^ 2
    jieta = value(meta) ^ 2 / (4*pi*fpi) ^ 2
    mpiL = value(mpi) * ens.L
    mkL = value(mk) * ens.L
    metaL = value(meta) * ens.L
    lampi = mpiL * sqrt.(nn)
    lamk = mkL * sqrt.(nn)
    lameta = metaL * sqrt.(nn)
    g1pi = sum(4 .* mm ./ lampi .* besselk.(1, lampi))
    g1k = sum(4 .* mm ./ lamk .* besselk.(1, lamk))
    g1eta = sum(4 .* mm ./ lameta .* besselk.(1, lameta))
    fve_mpi = 1/4 * jipi * g1pi - 1/12 * jieta * g1eta
    fve_mk = 1/6 * jieta * g1eta
    fve_fpi = - jipi * g1pi - 1/2 * jik * g1k
    fve_fk = -3/8 * jipi *g1pi - 3/4 * jik * g1k - 3/8 * jieta * g1eta

    mpi_infty = mpi / (1+fve_mpi)
    mk_infty = mk / (1+fve_mk)
    fpi_infty = fpi / (1+fve_fpi)
    fk_infty = fk / (1+fve_fk)

    return mpi_infty, mk_infty, fpi_infty, fk_infty
end

