@doc raw"""
    function plot_data(data::Vector{uwreal},xdata::Union{Vector{uwreal},Vector{T} where T<:real};
      wpm::Union{Dict{String, Vector{Float64}},Nothing} = nothing,
      c::String = "k",
      fmt::String = "x",
      label::AbstractString = "",
      title::AbstractString = "",
      xlabel::AbstractString = "x",
      ylabel::AbstractString = "y",
      figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing)

Generate a plot with `data` as function of `xdata`. If `xdata` is a `Vector{uwreal}`
it plots the error bar also in the x-axis. 
It return a tuple `fig,ax` to easily allow any modification to the plot. 

# Arguments  
- `wpm::Union{Dict{String,Vector{Float64}},Dict{Int64,Vector{Float64}},Nothing}=nothing`
  windows parameter used by `ADerrors` to estimate the integration window
- `label::AbstractString = ""`: set the label of `data`. When set, it prints the legend
- `figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing`: 
  A tuple with an exiting `fig` and `ax`. If passed, the fuction prints the data on
  the existing figure

# Examples
```@example
  using ADerror, juobs

  xdata = collect(1:100)
  ydata = [uwreal([sin(x*0.1*π),0.01],"sin") for x in xdata]
  fig,ax = data_plot(ydata,xdata,label=L"$\sin(\frac{x*\pi}{10}$")
  ydata2 = [uwreal([cos(x*0.1*π),0.01],"cos") for x in xdata]
  xdata2 = [uwreal([x,0.1], "xdata") for x in xdata];
  fig,ax = data_plot(ydata2,xdata2,label=L"$\cos(\frac{x*\pi}{10}$",figs = (fig,ax))
```
See also: [`plot_func`](@ref)
"""
function plot_data(data::Vector{uwreal},xdata::Vector{T} where T <: Real; 
  wpm::Union{Dict{String, Vector{Float64}},Nothing} = nothing,
  c = "k",
  fmt = "x",
  label::AbstractString = "",
  title::AbstractString = "",
  xlabel::AbstractString = "x",
  ylabel::AbstractString = "y",
  figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing)

  isnothing(wpm) ? uwerr.(data) : [uwerr(d,wpm) for d in data];
  fig, ax = isnothing(figs) ?  subplots(1) : figs;
  line, = ax.errorbar(xdata,value.(data), err.(data), c=c,fmt=fmt)
  if length(label)!=0
    line.set_label(label)
    ax.legend();
  end
  fig.suptitle(title);
  ax.set_ylabel(ylabel);
  ax.set_xlabel(xlabel);
  return fig,ax;
end

function plot_data(data::Vector{uwreal},xdata::Vector{uwreal}; 
  wpm::Union{Dict{String, Vector{Float64}},Nothing} = nothing,
  c = "k",
  fmt = "x",
  label::AbstractString = "",
  title::AbstractString = "",
  xlabel::AbstractString = "x",
  ylabel::AbstractString = "y",
  figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing)

  isnothing(wpm) ? uwerr.(data)  : [uwerr(d,wpm) for d in data];
  isnothing(wpm) ? uwerr.(xdata) : [uwerr(x,wpm) for x in xdata];
  fig, ax = isnothing(figs) ?  subplots(1) : figs;
  line, = ax.errorbar(value.(xdata),value.(data), ADerrors.err.(data), ADerrors.err.(xdata),c=c,fmt=fmt)
  if length(label)!=0
    line.set_label(label)
    ax.legend();
  end
  fig.suptitle(title);
  ax.set_ylabel(ylabel);
  ax.set_xlabel(xlabel);
  return fig,ax;
end

@doc raw"""
    function plot_func(f::Function,xdata::Vector{T} where T<:Real;
      wpm::Union{Dict{String,Vector{Float64}},Nothing} = nothing,
      c::String ="green",
      alpha::Float64=0.25,
      label::AbstractString = "",
      figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing,
It plots the function `f` evaluated at `xdata`through broadcasting. 
The function assumes that `f` return type is `uwreal` and is called as `f(x)`.
It returns a tuple with `fig,ax` to easiy edit the plot

# Arguments  
- `wpm::Union{Dict{String,Vector{Float64}},Dict{Int64,Vector{Float64}},Nothing}=nothing`
  windows parameter used by `ADerrors` to estimate the integration window
- `label::AbstractString = ""`: set the label of `data`. When set, it prints the legend
- `figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing`: 
  A tuple with an exiting `fig` and `ax`. If passed, the fuction prints the data on
  the existing figure

# Example
```@example
  using ADerror, juobs

  xdata = collect(1:100)
  f(x) = uwreal([sin(x*0.1*π),0.1],"sin")
  fig,ax = plot_func(f,xdata,label=L"\sin(\frac{x*\pi}{10})")
```
See also: [`plot_data`](@ref)
"""
function plot_func(f::Function, xdata::Vector{T} where T<:Real; 
  wpm::Union{Dict{String,Vector{Float64}},Nothing} = nothing,
  c::String ="green",
  alpha::Float64=0.25,
  figs::Union{Tuple{Figure, PyPlot.PyCall.PyObject},Nothing}=nothing,
  label::AbstractString = "",
  kwargs...) 
  fig,ax = isnothing(figs) ? subplots(1) : figs
  y = f.(xdata);
  isnothing(wpm) ? uwerr.(y) : [uwerr(a,wpm) for a in y];
  v,e = value.(y), err.(y);
  l = ax.fill_between(xdata,v.-e,v.+e, color=c,alpha=alpha);
  if label!=""
    l.set_label(label);
    ax.legend();
  end
  v,i = findmax(x->value(x), y);
  e = err(y[i])
  ax.set_ylim(top=v+20*e)
  return fig,ax;
end